
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2026 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  Modified versions of this work may be distributed,
// but only under the same license.  For further details, see the
// Creative Commons License "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

#include "kiosk/runnable.hh"
#include "kiosk/graph.hh"
#include "kiosk/displays.hh"

#include "graph_2.hh"
#include "languages.hh"
#include "demo_app.hh"

class graph_description : public window
{
public:
  void draw_dirty();
  
  graph_description() : window(BLACK) {}
  virtual ~graph_description() {}
};


void graph_description::draw_dirty()
{
  set_text_size(height/15);

  static bool init = true;
  
  static multilingua graphing_demo;
  if(init)
    {
      graphing_demo.set_text("Graphing Demo", ENGLISH);
      graphing_demo.set_text("绘图演示", CHINESE_SIMPLE);
      graphing_demo.set_text("圖形示範", CHINESE);
      graphing_demo.set_text("ग्राफिंग डेमो", HINDI);
      graphing_demo.set_text("Demostración de gráficos", SPANISH);
      graphing_demo.set_text("Démo de création de graphiques", FRENCH);
      graphing_demo.set_text("عرض توضيحي للرسوم البيانية", ARABIC);
      graphing_demo.set_text("গ্রাফিং ডেমো", BENGALI);
      graphing_demo.set_text("Демонстрація графіків", UKRANIAN);
      graphing_demo.set_text("Demonstração de Gráficos", PORTUGUESE);
      graphing_demo.set_text("گرافنگ ڈیمو", URDU);
      graphing_demo.set_text("Demo Pembuatan Grafik", INDONESIAN);
      graphing_demo.set_text("Grafik-Demo", GERMAN);
      graphing_demo.set_text("グラフ作成デモ", JAPANESE);
      graphing_demo.set_text("Επίδειξη γραφημάτων", GREEK);
      graphing_demo.set_text("Dimostrazione di creazione di grafici", ITALIAN);
      graphing_demo.set_text("Демо построения графиков", RUSSIAN);
      graphing_demo.set_text("הדגמת גרפים", HEBREW);
    }

  
  draw_text(graphing_demo,
	    WHITE,
	    width/2,
	    height*11/100,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  set_text_size(height/30);

  dt_colors textcolors = { GREEN7, REDRED8, WHITE };
  
  static multilingua back_text;
  if(init)
    {
      back_text.set_text("To go back to the menu, " DT_COLOR_1 "click or touch at the screen top left", ENGLISH);
      back_text.set_text("要返回菜单，" DT_COLOR_1 "请点击或触摸屏幕左上角。", CHINESE_SIMPLE);
      back_text.set_text("若要返回選單，" DT_COLOR_1 "請點選或觸碰螢幕左上角。", CHINESE);
      back_text.set_text("मेनू पर वापस जाने के लिए, स्क्रीन के ऊपर बाईं ओर क्लिक करें या टच करें।", HINDI);
      back_text.set_text("Para volver al menú, " DT_COLOR_1 "haz clic o toca la parte superior izquierda de la pantalla.", SPANISH);
      back_text.set_text("Pour revenir au menu, " DT_COLOR_1 "cliquez ou touchez la partie supérieure gauche de l'écran.", FRENCH);
      back_text.set_text(U"للعودة إلى القائمة، انقر أو المس الجزء العلوي الأيسر من الشاشة.", ARABIC);
      back_text.set_text("মেনুতে ফিরে যেতে, " DT_COLOR_1 "স্ক্রিনের উপরের বাম দিকে ক্লিক করুন বা স্পর্শ করুন।", BENGALI);
      back_text.set_text("Щоб повернутися до меню, " DT_COLOR_1 "натисніть або торкніться у верхньому лівому куті екрана", UKRANIAN);
      back_text.set_text("Para voltar ao menu, " DT_COLOR_1 "clique ou toque no canto superior esquerdo do ecrã.", PORTUGUESE);
      back_text.set_text("مینو پر واپس جانے کے لیے، اسکرین کے اوپر بائیں جانب کلک کریں یا ٹچ کریں۔", URDU);
      back_text.set_text("Untuk kembali ke menu, " DT_COLOR_1 "klik atau sentuh bagian kiri atas layar.", INDONESIAN);
      back_text.set_text("Um zum Menü zurückzukehren, " DT_COLOR_1 "klicken oder tippen Sie oben links auf dem Bildschirm.", GERMAN);
      back_text.set_text("メニューに戻るには、" DT_COLOR_1 "画面左上をクリックまたはタップしてください。", JAPANESE);
      back_text.set_text("Για να επιστρέψετε στο μενού, " DT_COLOR_1 "κάντε κλικ ή αγγίξτε το επάνω αριστερά στην οθόνη", GREEK);
      back_text.set_text("Per tornare al menu, " DT_COLOR_1 "clicca o tocca la parte in alto a sinistra dello schermo.", ITALIAN);
      back_text.set_text("Чтобы вернуться в меню, " DT_COLOR_1 "нажмите или коснитесь экрана в левом верхнем углу.", RUSSIAN);
      back_text.set_text("כדי לחזור לתפריט, לחצו או געו ב- בפינה השמאלית העליונה של המסך", HEBREW);
    }

  draw_multicolored_text(back_text,
                         0,
                         textcolors,
                         width/2,
                         height*25/100,
                         DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);


  static multilingua text1;
  if(init)
    {
      text1.set_text("Graphs on a dedicated screen maximize viewing area", ENGLISH);
      text1.set_text("图表显示在专用屏幕上，可最大限度地扩大观看区域。", CHINESE_SIMPLE);
      text1.set_text("在專用螢幕上顯示圖表，可最大程度地擴大可視區域。", CHINESE);
      text1.set_text("एक डेडिकेटेड स्क्रीन पर ग्राफ़ देखने के एरिया को ज़्यादा से ज़्यादा करते हैं।", HINDI);
      text1.set_text("Los gráficos en una pantalla dedicada maximizan el área de visualización.", SPANISH);
      text1.set_text("Les graphiques affichés sur un écran dédié maximisent la zone de visualisation.", FRENCH);
      text1.set_text("تساهم الرسوم البيانية المعروضة على شاشة مخصصة في زيادة مساحة العرض إلى أقصى حد.", ARABIC);
      text1.set_text("একটি ডেডিকেটেড স্ক্রিনে গ্রাফগুলো দেখার জন্য উপলব্ধ স্থানকে সর্বোচ্চ করে তোলে।", BENGALI);
      text1.set_text("Графіки на спеціальному екрані максимально збільшують область перегляду", UKRANIAN);
      text1.set_text("Os gráficos num ecrã dedicado maximizam a área de visualização.", PORTUGUESE);
      text1.set_text("ایک وقف شدہ اسکرین پر گراف دیکھنے کے علاقے کو زیادہ سے زیادہ بناتے ہیں۔", URDU);
      text1.set_text("Grafik pada layar khusus memaksimalkan area tampilan.", INDONESIAN);
      text1.set_text("Diagramme auf einem separaten Bildschirm maximieren die Anzeigefläche.", GERMAN);
      text1.set_text("専用画面にグラフを表示することで、表示領域を最大限に活用できます。", JAPANESE);
      text1.set_text("Τα γραφήματα σε μια ειδική οθόνη μεγιστοποιούν την περιοχή προβολής", GREEK);
      text1.set_text("I grafici visualizzati su uno schermo dedicato massimizzano l'area di visualizzazione.", ITALIAN);
      text1.set_text("Графики на отдельном экране обеспечивают максимальную площадь обзора.", RUSSIAN);
      text1.set_text("גרפים על מסך ייעודי ממקסמים את אזור הצפייה", HEBREW);
    }

  draw_text(text1,
	    GREY8,
	    width/2,
	    height*35/100,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua text2;
  if(init)
    {
      text2.set_text("Automatic tiling of graphs is demonstrated", ENGLISH);
      text2.set_text("演示了图表的自动平铺功能。", CHINESE_SIMPLE);
      text2.set_text("本文示範了圖的自動平鋪。", CHINESE);
      text2.set_text("ग्राफ़ की ऑटोमैटिक टाइलिंग का प्रदर्शन किया गया है।", HINDI);
      text2.set_text("Se demuestra la disposición automática de gráficos en mosaico.", SPANISH);
      text2.set_text("La disposition automatique des graphiques est présentée.", FRENCH);
      text2.set_text("تم عرض عملية التبليط التلقائي للرسوم البيانية.", ARABIC);
      text2.set_text("গ্রাফের স্বয়ংক্রিয় টাইলস বিন্যাস প্রদর্শন করা হয়েছে।", BENGALI);
      text2.set_text("Демонстровано автоматичне мозаїчне розбиття графів", UKRANIAN);
      text2.set_text("É demonstrada a divisão automática de gráficos em mosaicos.", PORTUGUESE);
      text2.set_text("گراف کی خودکار ٹائلنگ کا مظاہرہ کیا گیا ہے۔", URDU);
      text2.set_text("Penataan ubin grafik secara otomatis ditunjukkan.", INDONESIAN);
      text2.set_text("Die automatische Kachelung von Graphen wird demonstriert.", GERMAN);
      text2.set_text("グラフの自動タイル分割のデモンストレーションを行います。", JAPANESE);
      text2.set_text("Παρουσιάζεται η αυτόματη τοποθέτηση πλακιδίων σε γραφήματα", GREEK);
      text2.set_text("Viene dimostrata la disposizione automatica dei grafici in un layout a griglia.", ITALIAN);
      text2.set_text("Демонстрируется автоматическое размещение графиков на плоскости.", RUSSIAN);
      text2.set_text("מוצג ריצוף אוטומטי של גרפים", HEBREW);
    }

  draw_text(text2,
	    GREY8,
	    width/2,
	    height*45/100,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua text3;
  if(init)
    {
      text3.set_text("Graphs have different colors and text sizes", ENGLISH);
      text3.set_text("图表使用不同的颜色和文字大小。", CHINESE_SIMPLE);
      text3.set_text("圖表具有不同的顏色和文字大小。", CHINESE);
      text3.set_text("ग्राफ में अलग-अलग रंग और टेक्स्ट साइज़ होते हैं।", HINDI);
      text3.set_text("Los gráficos tienen diferentes colores y tamaños de texto.", SPANISH);
      text3.set_text("Les graphiques présentent des couleurs et des tailles de texte différentes.", FRENCH);
      text3.set_text("تتميز الرسوم البيانية بألوان وأحجام نصوص مختلفة.", ARABIC);
      text3.set_text("গ্রাফগুলিতে বিভিন্ন রঙ এবং লেখার আকার থাকে।", BENGALI);
      text3.set_text("Графіки мають різні кольори та розміри тексту", UKRANIAN);
      text3.set_text("Os gráficos têm cores e tamanhos de texto diferentes.", PORTUGUESE);
      text3.set_text("گراف میں مختلف رنگ اور متن کے سائز ہوتے ہیں۔", URDU);
      text3.set_text("Grafik-grafik tersebut memiliki warna dan ukuran teks yang berbeda.", INDONESIAN);
      text3.set_text("Die Diagramme haben unterschiedliche Farben und Textgrößen.", GERMAN);
      text3.set_text("グラフは色や文字サイズがそれぞれ異なります。", JAPANESE);
      text3.set_text("Τα γραφήματα έχουν διαφορετικά χρώματα και μεγέθη κειμένου", GREEK);
      text3.set_text("I grafici presentano colori e dimensioni del testo diversi.", ITALIAN);
      text3.set_text("Графики имеют разные цвета и размеры текста.", RUSSIAN);
      text3.set_text("לגרפים יש צבעים וגדלי טקסט שונים", HEBREW);
    }

  draw_text(text3,
	    GREY8,
	    width/2,
	    height*55/100,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua text4;
  if(init)
    {
      text4.set_text("One graph has user axis scale changes disabled", ENGLISH);
      text4.set_text("其中一个图表禁用了用户更改坐标轴刻度尺度的功能。", CHINESE_SIMPLE);
      text4.set_text("其中一張圖表已停用使用者軸刻度變更功能。", CHINESE);
      text4.set_text("एक ग्राफ़ में यूज़र एक्सिस स्केल बदलने की सुविधा डिसेबल है।", HINDI);
      text4.set_text("En uno de los gráficos, se han deshabilitado los cambios en la escala del eje del usuario.", SPANISH);
      text4.set_text("Sur ce graphique, la modification de l'échelle des axes par l'utilisateur est désactivée.", FRENCH);
      text4.set_text("تم تعطيل إمكانية تغيير مقياس المحور من قبل المستخدم في أحد الرسوم البيانية.", ARABIC);
      text4.set_text("একটি গ্রাফে ব্যবহারকারী কর্তৃক অক্ষের স্কেল পরিবর্তন নিষ্ক্রিয় করা আছে।", BENGALI);
      text4.set_text("На одному графіку вимкнено зміну масштабу осі користувача", UKRANIAN);
      text4.set_text("Um dos gráficos tem as alterações de escala do eixo definidas pelo utilizador.", PORTUGUESE);
      text4.set_text("ایک گراف میں صارف کے محور پیمانے کی تبدیلیاں غیر فعال ہیں۔", URDU);
      text4.set_text("Salah satu grafik memiliki fitur perubahan skala sumbu pengguna yang dinonaktifkan.", INDONESIAN);
      text4.set_text("Bei einem Diagramm sind die Änderungen der Achsenskalierung durch den Benutzer deaktiviert.", GERMAN);
      text4.set_text("あるグラフでは、ユーザーによる軸目盛りの変更が無効になっています。", JAPANESE);
      text4.set_text("Ένα γράφημα έχει απενεργοποιημένες τις αλλαγές κλίμακας άξονα χρήστη", GREEK);
      text4.set_text("In un grafico, la modifica della scala dell'asse utente è disabilitata.", ITALIAN);
      text4.set_text("На одном из графиков отключена возможность изменения масштаба оси пользователем.", RUSSIAN);
      text4.set_text("בגרף אחד שינויי קנה מידה של ציר המשתמש מושבתים", HEBREW);
    }

  draw_text(text4,
	    GREY8,
	    width/2,
	    height*65/100,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua text5;
  if(init)
    {
      text5.set_text("Other graphs have scaling only for X or only for Y", ENGLISH);
      text5.set_text("其他图表仅对X轴或仅对Y轴进行缩放。", CHINESE_SIMPLE);
      text5.set_text("其他圖表僅對 X 軸或僅對 Y 軸進行刻度。", CHINESE);
      text5.set_text("अन्य ग्राफ़ में स्केलिंग केवल X के लिए या केवल Y के लिए होती है।", HINDI);
      text5.set_text("Otros gráficos tienen escalado solo para el eje X o solo para el eje Y.", SPANISH);
      text5.set_text("D'autres graphiques n'ont une mise à l'échelle que pour l'axe X ou seulement pour l'axe Y.", FRENCH);
      text5.set_text("تحتوي بعض الرسوم البيانية الأخرى على مقياس رسم للمحور X فقط أو للمحور Y فقط.", ARABIC);
      text5.set_text("অন্যান্য গ্রাফগুলিতে শুধুমাত্র X অক্ষের জন্য বা শুধুমাত্র Y অক্ষের জন্য স্কেলিং থাকে।", BENGALI);
      text5.set_text("Інші графіки мають масштабування лише для X або лише для Y", UKRANIAN);
      text5.set_text("Outros gráficos têm escala apenas para X ou apenas para Y.", PORTUGUESE);
      text5.set_text("دوسرے گراف میں صرف X یا صرف Y کے لیے اسکیلنگ ہوتی ہے۔", URDU);
      text5.set_text("Grafik lain hanya memiliki penskalaan untuk sumbu X atau hanya untuk sumbu Y.", INDONESIAN);
      text5.set_text("Andere Diagramme weisen eine Skalierung nur für die X-Achse oder nur für die Y-Achse auf.", GERMAN);
      text5.set_text("他のグラフは、X軸のみ、またはY軸のみに目盛りが付けられています。", JAPANESE);
      text5.set_text("Άλλα γραφήματα έχουν κλιμάκωση μόνο για X ή μόνο για Y", GREEK);
      text5.set_text("Altri grafici presentano una scala solo per l'asse X o solo per l'asse Y.", ITALIAN);
      text5.set_text("На других графиках масштабирование применяется только к оси X или только к оси Y.", RUSSIAN);
      text5.set_text("Other graphs have scaling only for X or only for Y", HEBREW);
    }

  draw_text(text5,
	    GREY8,
	    width/2,
	    height*75/100,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  init = false;
}

void data_update_main_2(void* data);

void switch_to_graph_2(class displays* d)
{
  //printf("Switching to graph2_pane.\n");
  d->set_current_pane(GRAPH2_PANE);
}

void switch_from_graph_2(class displays* d)
{
  d->set_current_pane(MENU_PANE);
}

bool my_hotfunc(hotspot* h, my_event& me, void* data)
{
  bool mouse_click = me.source_mouse && me.type==EVENT_TOUCH && me.mouse_buttons_pressed==MOUSE_BUTTON_LEFT;

  bool touched = (!me.source_mouse) && me.type==EVENT_TOUCH && me.num_touches==1;

  if(mouse_click || touched)  
    {
      displays* d = (displays*)data;
      d->touch_recognized_beep();
      switch_from_graph_2(d);
      h->trash_events_until_release();
      return true;
    }
  
  return false;
}

//
// The purpose of having a derived class here is simply to customize
// the layout function to place the hotspot.
//
class graph_2_window : public multiwindow
{
public:
  hotspot* hsp = 0;
  virtual void layout();
  graph_2_window() {}
  virtual ~graph_2_window() {}
};

void graph_2_window::layout()
{
  if(hsp)
    hsp->resize(0*width/100, 0*height/100, 6*width/100, 10*height/100);
    
  multiwindow::layout();  
}


multiwindow* setup_graph_2(displays* d)
{
  printf("Setting up graph 2.\n");

  graph_2_window* mw = new graph_2_window();
  
  d->add(mw, GRAPH2_PANE);
  
  graph** gr = new graph*[3];
  gr[0] = new graph;
  gr[1] = new graph;
  gr[2] = new graph;
  graph_description* gd = new graph_description;

  mw->add(gr[0]);
  mw->add(gr[1]);
  mw->add(gr[2]);
  mw->add(gd);

  // Last window added is the first to get events.
  mw->hsp = new hotspot(my_hotfunc, (void*)d, true);
  mw->add(mw->hsp);


  
  for(int g=0; g<3; g++)
    {
      gr[g]->set_x_axis_title(U"X Axis",    ENGLISH);
      gr[g]->set_x_axis_title(U"X轴",       CHINESE_SIMPLE);
      gr[g]->set_x_axis_title(U"X軸",       CHINESE);
      gr[g]->set_x_axis_title(U"एक्स अक्ष",   HINDI);
      gr[g]->set_x_axis_title(U"Eje X",     SPANISH);
      gr[g]->set_x_axis_title(U"Axe X",     FRENCH);
      gr[g]->set_x_axis_title(U"المحور X",   ARABIC);
      gr[g]->set_x_axis_title(U"এক্স অক্ষ",   BENGALI);
      gr[g]->set_x_axis_title(U"Вісь X",    UKRANIAN);
      gr[g]->set_x_axis_title(U"Eixo X",    PORTUGUESE);
      gr[g]->set_x_axis_title(U"ایکس محور", URDU);
      gr[g]->set_x_axis_title(U"Sumbu X",   INDONESIAN);
      gr[g]->set_x_axis_title(U"X-Achse",   GERMAN);
      gr[g]->set_x_axis_title(U"X軸",       JAPANESE);
      gr[g]->set_x_axis_title(U"Άξονας Χ",  GREEK);
      gr[g]->set_x_axis_title(U"Asse X",    ITALIAN);
      gr[g]->set_x_axis_title(U"Ось X",     RUSSIAN);
      gr[g]->set_x_axis_title(U"ציר X",      HEBREW);
      
      gr[g]->set_y_axis_title(U"Y Axis",    ENGLISH);
      gr[g]->set_y_axis_title(U"Y轴",       CHINESE_SIMPLE);
      gr[g]->set_y_axis_title(U"Y軸",       CHINESE);
      gr[g]->set_y_axis_title(U"Y अक्ष",     HINDI);
      gr[g]->set_y_axis_title(U"Eje Y",     SPANISH);
      gr[g]->set_y_axis_title(U"Axe Y",     FRENCH);
      gr[g]->set_y_axis_title(U"المحور Y",   ARABIC);
      gr[g]->set_y_axis_title(U"Y অক্ষ",    BENGALI);
      gr[g]->set_y_axis_title(U"Вісь Y",    UKRANIAN);
      gr[g]->set_y_axis_title(U"Eixo Y",    PORTUGUESE);
      gr[g]->set_y_axis_title(U"Y ایکسس",   URDU);
      gr[g]->set_y_axis_title(U"Sumbu Y",   INDONESIAN);
      gr[g]->set_y_axis_title(U"Y-Achse",   GERMAN);
      gr[g]->set_y_axis_title(U"Y軸",       JAPANESE);
      gr[g]->set_y_axis_title(U"Άξονας Y",  GREEK);
      gr[g]->set_y_axis_title(U"Asse Y",    ITALIAN);
      gr[g]->set_y_axis_title(U"Ось Y",     RUSSIAN);
      gr[g]->set_y_axis_title(U"ציר Y",     HEBREW);
      
      gr[g]->set_title(U"English Graph Title 😃",             ENGLISH);
      gr[g]->set_title(U"简体中文图形标题 😃",                CHINESE_SIMPLE);
      gr[g]->set_title(U"中文圖名 😃",                        CHINESE);
      gr[g]->set_title(U"हिंदी ग्राफ शीर्षक 😃",                   HINDI);
      gr[g]->set_title(U"Título del gráfico en español 😃",   SPANISH);
      gr[g]->set_title(U"Titre du graphique en français 😃",  FRENCH);
      gr[g]->set_title(U"عنوان الرسم البياني باللغة العربية 😃",       ARABIC);
      gr[g]->set_title(U"বাংলা গ্রাফ শিরোনাম 😃",               BENGALI);
      gr[g]->set_title(U"Назва українського графіка 😃",      UKRANIAN);
      gr[g]->set_title(U"Título do gráfico em português 😃",  PORTUGUESE);
      gr[g]->set_title(U"اردو گراف کا عنوان 😃",                 URDU);
      gr[g]->set_title(U"Judul Grafik Bahasa Indonesia 😃",   INDONESIAN);
      gr[g]->set_title(U"Deutscher Graphtitel 😃",            GERMAN);
      gr[g]->set_title(U"日本語グラフタイトル 😃",            JAPANESE);
      gr[g]->set_title(U"Τίτλος ελληνικού γραφήματος 😃",     GREEK);
      gr[g]->set_title(U"Titolo del grafico italiano 😃",     ITALIAN);
      gr[g]->set_title(U"Русский Граф Заголовок 😃",          RUSSIAN);
      gr[g]->set_title(U"כותרת גרף עברי 😃",                  HEBREW);
    }

  gr[1]->set_background_color  (GREY5);
  gr[1]->set_title_color       (BLACK);
  gr[1]->set_panel_color       (GREY7);
  gr[1]->set_axis_color        (BLACK);
  gr[1]->set_axis_title_color  (BLACK);
  gr[1]->set_tic_label_color   (BLACK);
  gr[1]->set_tic_color         (BLACK);
  gr[1]->set_grid_major_color  (GREY8);
  gr[1]->set_grid_minor_color  (GREY3);
  gr[1]->set_title_font_height         (  8, 25, 1000 );
  gr[1]->set_axis_title_font_height    (  7, 20, 1000 );
  gr[1]->set_tic_label_font_height     (  5,  6, 1000 );
  gr[1]->set_dataset_label_font_height (  5, 15, 1000 );
  gr[1]->set_error_text_font_height    (  12, 25, 1000 );

  gr[2]->set_background_color(GREY3);
  gr[2]->set_title_color(BLUE8);
  gr[2]->set_panel_color(BLUE1);
  gr[2]->set_title_font_height         (  4, 30, 1000 );
  gr[2]->set_axis_title_font_height    (  3, 25, 1000 );
  gr[2]->set_tic_label_font_height     (  2, 20, 1000 );
  gr[2]->set_dataset_label_font_height (  2, 15, 1000 );
  gr[2]->set_error_text_font_height    (  6, 50, 1000 );

  gr[0]->allow_user_resize_x(false);
  gr[0]->allow_user_resize_y(false);

  gr[1]->allow_user_resize_x(false);

  gr[2]->allow_user_resize_y(false);

  new runnable(data_update_main_2, gr);

  return mw;
}

static double sinc(double x)
{
  if(x==0.0)
    return 1.0;
  
  return sin(x)/x;
}

void data_update_main_2(void* data)
{
  graph** gr = (graph**) data;

  int num_graph_points = 201;

  graph_data* gd1[3];
  graph_data* gd2[3];
  graph_data* gd[3];

  for(int g=0; g<3; g++)
    {
      gd1[g] = new graph_data(1,                         // max_datasets
			      num_graph_points,          // num_points
			      -1.0,                      // x_data_start
			      2.0/(num_graph_points-1)); // x_data_step

      gd2[g] = new graph_data(1,                         // max_datasets
			      num_graph_points,          // num_points
			      -1.0,                      // x_data_start
			      2.0/(num_graph_points-1)); // x_data_step

      gd[g] = gd1[g];
    }
  
  
  printf("**** Started update for graph_2\n");

  int direction = 1;

  int n = 0;
  
  for(;;)
    {
      usleep(10000);

      // Note that if the graphs aren't currently being displayed,
      // none will be ready for new data.  So minimal CPU is spent here.
      
      for(int g=0; g<3; g++)
	{
	  if(gr[g]->ready_for_new_data())
	    {
	      gd[g]->x_data_start = -4;
	      float x_data_end = 5;
	      gd[g]->x_data_step  = (x_data_end - gd[g]->x_data_start) / (gd[g]->num_points-1);
	      
	      for(int i=0; i<gd[g]->num_points; i++)
		{
		  double x = 2.0 * i / (gd[g]->num_points-1) - 1.0;
		  double a = sinc(10.0*x);
		  double b = x;
		  int m = (n+50)>100?150-n:n+50;
		  if(g==0)
		    gd[g]->y_data[0]->data[i] = (n*a + (100-n)*b)/100;
		  else if(g==1)
		    gd[g]->y_data[0]->data[i] = (n*b + (100-n)*a)/100;
		  else
		    gd[g]->y_data[0]->data[gd[g]->num_points-1-i] = (m*b + (100-m)*a)/100;
		}

	      // printf("Supplied graph_data.\n");
	      gr[g]->accept_data(gd[g]);
      
	      if(gd[g]==gd1[g])
		gd[g] = gd2[g];
	      else
		gd[g] = gd1[g];
	    }
	}
      
      if(n==0)
	direction = 1;
      else if(n==100)
	direction = -1;
      
      n+=direction;
    }
}
