
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2026 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  Modified versions of this work may be distributed,
// but only under the same license.  For further details, see the
// Creative Commons License "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

#include <locale.h>

#include "kiosk/select_one.hh"

#include "languages.hh"

const char32_t* languages[] =
  {
    U"English",               // 0
    U"Chinese 中国人",        // 1
    U"Chinese 中國人",        // 2
    U"Hindi हिंदी",             // 3
    U"Spanish Español",       // 4
    U"French Français",       // 5
    U"Arabic عربي",           //
    U"Bengali বাংলা",         // 7
    U"Ukranian українська",   // 8
    U"Portuguese Português",  // 9
    U"Urdu اردو",              //
    U"Indonesian Indonesia",  // 11
    U"German Deutsch",        // 12
    U"Japanese 日本語",       // 13
    U"Greek ελληνικά",        // 14
    U"Italian Italiana",      // 15
    U"Russian Русский",       // 16
    U"Hebrew עִברִית",          //
    0
  };

void language_change_function(selection_changed_event* sce, multiselect* ms, void* data)
{
  display* d = ms->get_display();

  printf("Setting language number to %d\n", sce->selection_number);
  
  if(sce->is_selected)
    multilingua::set_language(d, sce->selection_number);
}


select_one* setup_language_select()
{
  multilingua title;
  title.set_text("Language", ENGLISH);
  select_one* sel1 = new select_one(title);

  for(int i=0; languages[i]; i++)
    {
      multilingua name;
      name.set_text(languages[i], ENGLISH);
      sel1->add(name, i==0);
    }
  
  sel1->change_function_data = 0;
  sel1->change_function      = language_change_function;

  return sel1;
}


//
// It appears that one can change printf output to produce commas
// using the locale.  However, that requires many things outside
// this program to be set up and working correctly.  So instead,
// we'll specify the "C" locale, which always produces decimal
// points.  We'll then manually convert them to commas when
// required.  This is perhaps ugly, but not so ugly as using
// locale, which appears to be entirely system-dependent and
// unreliable for this task.
//
void decimal_change_function(selection_changed_event* sce, multiselect* ms, void* data)
{
  display* d = ms->get_display();

  printf("Setting decimal format to \"%s\"\n", sce->selection_number?"3.1415927":"3,1415927");
  
  if(sce->is_selected)
    {
      if(sce->selection_number==0)
	use_comma_for_decimal_point = false;
      else	
	use_comma_for_decimal_point = true;
      
      d->mark_layout_dirty();
    }
}


select_one* setup_decimal_select()
{
  setlocale(LC_NUMERIC, "C");

  multilingua txt;
  txt.set_text("Decimals", ENGLISH);

  select_one* sel1 = new select_one(txt);

  txt.set_text("3.1415927", ENGLISH);
  sel1->add(txt, true);

  txt.set_text("3,1415927", ENGLISH);
  sel1->add(txt, false);

  sel1->change_function_data = 0;
  sel1->change_function      = decimal_change_function;

  return sel1;
}


