
// SPDX-License-Identifier: CC-BY-NC-SA-4.0
//
// Copyright (C) 2026 Bit by Bit Signal Processing LLC (https://bxbsp.com)
//
// This work is placed under the "Creative Commons Attribution
// NonCommercial ShareAlike 4.0 International" license, known
// by the shortened acronym "CC-BY-NC-SA-4.0".
//
// This work is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// A CC-BY-NC-SA-4.0 license allows you to use this work for
// noncommercial purposes so long as attribution is made to the
// original author.  Modified versions of this work may be distributed,
// but only under the same license.  For further details, see the
// Creative Commons License "CC-BY-NC-SA-4.0".
//
// You should have received a copy of the CC-BY-NC-SA-4.0 license
// along with this work. If not, see
// <https://creativecommons.org/licenses/by-nc-sa/4.0/>.
//

#include "kiosk/window.hh"
#include "kiosk/multiwindow.hh"
#include "kiosk/button.hh"
#include "kiosk/unalingua.hh"
#include "kiosk/selection.hh"

#include "selection_demo.hh"
#include "languages.hh"

void button_push_function_questionable_action(button*b, void* v);
void button_push_function_favorite_color(button*b, void* v);

#define NUM_FAVS 6
multilingua* favorite_colors = 0;
dt_colors fav_colors = { WHITE, RED, GREEN, BLUE, YELLOW, ORANGE, PURPLE };

class selection_demo_window : public multiwindow
{
  button* b;
  button* favorite_color_b;

  virtual void layout();
  virtual void draw_dirty();
  virtual void draw_dynamic();

public:

  const static int text_display_interval    = 100;

  unalingua        last_text;
  int              last_text_count;

  selection_demo_window();
  ~selection_demo_window();
};

window* create_selection_demo_window()
{
  return new selection_demo_window();
}


selection_demo_window::~selection_demo_window()
{
  delete b;
  delete favorite_color_b;
}

selection_demo_window::selection_demo_window()
{
  last_text_count = 0;

  dt_colors c  = { WHITE };
  dt_colors cf = { RED };

  multilingua b_name;
  b_name.set_text("Push to test a questionable action", ENGLISH);
  b_name.set_text(U"尝试执行一项可疑的操作以进行测试。", CHINESE_SIMPLE);
  b_name.set_text(U"推動測試一項可疑行為", CHINESE);
  b_name.set_text(U"एक संदिग्ध कार्रवाई का परीक्षण करने के लिए पुश करें", HINDI);
  b_name.set_text(U"Presionar para poner a prueba una acción cuestionable.", SPANISH);
  b_name.set_text(U"Insister pour tester une action douteuse.", FRENCH);
  b_name.set_text(U"الضغط من أجل اختبار إجراء مشكوك فيه.", ARABIC);
  b_name.set_text(U"একটি প্রশ্নবিদ্ধ পদক্ষেপ পরীক্ষা করার জন্য চাপ দেওয়া", BENGALI);
  b_name.set_text(U"Натискання для перевірки сумнівної дії", UKRANIAN);
  b_name.set_text(U"Insistir para testar uma ação questionável", PORTUGUESE);
  b_name.set_text(U"قابل اعتراض کارروائی کو جانچنے کے لیے دبائیں", URDU);
  b_name.set_text(U"Dorong untuk menguji tindakan yang meragukan.", INDONESIAN);
  b_name.set_text(U"Drängen Sie darauf, die fragwürdige Handlung zu überprüfen.", GERMAN);
  b_name.set_text(U"疑わしい行動を検証するために、テストを実施するよう働きかける", JAPANESE);
  b_name.set_text(U"Πιέστε για να δοκιμάσετε μια αμφισβητήσιμη ενέργεια", GREEK);
  b_name.set_text(U"Insistere per testare un'azione discutibile.", ITALIAN);
  b_name.set_text(U"Настаивайте на проверке сомнительного действия.", RUSSIAN);
  b_name.set_text(U"דחיפה לבדיקת פעולה מפוקפקת", HEBREW);

  b = new button(b_name,
		 5,       //  x_margin_percent
		 5,       //  y_margin_percent
		 20,      //  circle_width_percent
		 c,       //  textcolor
		 GREY2,   //  fg
		 cf,      //  flash textcolor
		 BLUE,    //  flash bg
		 button_push_function_questionable_action,  // Function to call when pushed
		 (void*) this,   // data for function
		 true,    // tighten x
		 false);

  b->enable_flash = false;
  add(b);

  multilingua favorite_color_name;
  favorite_color_name.set_text("Push to select your favorite colour", ENGLISH);
  favorite_color_name.set_text(U"点击选择您喜欢的颜色", CHINESE_SIMPLE);
  favorite_color_name.set_text(U"按下按鈕選擇您喜歡的顏色", CHINESE);
  favorite_color_name.set_text(U"अपना पसंदीदा रंग चुनने के लिए दबाएँ", HINDI);
  favorite_color_name.set_text(U"Presiona para seleccionar tu color favorito.", SPANISH);
  favorite_color_name.set_text(U"Appuyez pour sélectionner votre couleur préférée.", FRENCH);
  favorite_color_name.set_text(U"اضغط لاختيار لونك المفضل.", ARABIC);
  favorite_color_name.set_text(U"আপনার পছন্দের রঙটি নির্বাচন করতে চাপ দিন।", BENGALI);
  favorite_color_name.set_text(U"Натисніть, щоб вибрати свій улюблений колір", UKRANIAN);
  favorite_color_name.set_text(U"Prima para selecionar a sua cor favorita.", PORTUGUESE);
  favorite_color_name.set_text(U"اپنے پسندیدہ رنگ کو منتخب کرنے کے لیے دبائیں", URDU);
  favorite_color_name.set_text(U"Tekan untuk memilih warna favorit Anda.", INDONESIAN);
  favorite_color_name.set_text(U"Drücken Sie, um Ihre Lieblingsfarbe auszuwählen.", GERMAN);
  favorite_color_name.set_text(U"お気に入りの色を選択するには、ボタンを押してください。", JAPANESE);
  favorite_color_name.set_text(U"Πιέστε για να επιλέξετε το αγαπημένο σας χρώμα", GREEK);
  favorite_color_name.set_text(U"Premi per selezionare il tuo colore preferito.", ITALIAN);
  favorite_color_name.set_text(U"Нажмите, чтобы выбрать свой любимый цвет.", RUSSIAN);
  favorite_color_name.set_text(U"לחץ כדי לבחור את הצבע המועדף עליך", HEBREW);

  favorite_color_b = new button(favorite_color_name,
				5,       //  x_margin_percent
				5,       //  y_margin_percent
				20,      //  circle_width_percent
				fav_colors,    //  textcolor
				GREY2,   //  fg
				cf,      //  flash textcolor
				BLUE,    //  flash bg
				button_push_function_favorite_color,  // Function to call when pushed
				(void*) this,  // data for function
				true,   // tighten x
				false);

  favorite_color_b->enable_flash = false;

  add(favorite_color_b);
}

void selection_demo_window::layout()
{
  //b->resize(20*width/100, 60*height/100, 60*width/100, 20*height/100);
  b->resize(0, 54*height/100, width, height/20);
  b->layout();

  favorite_color_b->resize(0, 20*height/100, width, height/20);
  favorite_color_b->layout();

  layout_dirty = false;
  dirty = true;
}

void selection_demo_window::draw_dynamic()
{
  //clear(GREY0);

  if(last_text_count)
    {
      last_text_count--;

      color c = scale(WHITE, BLACK, last_text_count / (float) text_display_interval);
      
      set_text_size(height/40);

      //draw_line(point(0, height*60/100), point(width-1, height*60/100), RED);
      
      draw_text(last_text,
		c,
		width/2,
		height*60/100,
		DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);
    }
  

  multiwindow::draw_dynamic();
}

void selection_demo_window::draw_dirty()
{
  multiwindow::draw_dirty();
    
  point p0(0,0);
  point p1(width-1, 0);
  point p2(width-1, height-1);
  point p3(0, height-1);

  draw_line_vertical(p1, p2, GREY7);
  draw_line_vertical(p0, p3, GREY7);
  draw_line_horizontal(p0, p1, GREY7);
  draw_line_horizontal(p2, p3, GREY7);
  
  set_text_size(height/25);

  static bool init = true;

  static multilingua selection_demo;
  if(init)
    {
      selection_demo.set_text("Ask Selection Demo", ENGLISH);
      selection_demo.set_text(U"选择询问演示", CHINESE_SIMPLE);
      selection_demo.set_text(U"Ask_Selection 演示", CHINESE);
      selection_demo.set_text(U"आस्क_सिलेक्शन डेमो", HINDI);
      selection_demo.set_text(U"Demostración de Ask_Selection", SPANISH);
      selection_demo.set_text(U"Démo de la fonction Ask_Selection", FRENCH);
      selection_demo.set_text(U"عرض توضيحي لـ Ask_Selection", ARABIC);
      selection_demo.set_text(U"আস্ক_সিলেকশন ডেমো", BENGALI);
      selection_demo.set_text(U"Демо Ask_Selection", UKRANIAN);
      selection_demo.set_text(U"Demonstração de Ask_Selection", PORTUGUESE);
      selection_demo.set_text(U"Ask_Select Demo", URDU);
      selection_demo.set_text(U"Demo Pemilihan Pertanyaan", INDONESIAN);
      selection_demo.set_text(U"Ask_Selection-Demo", GERMAN);
      selection_demo.set_text(U"選択デモを依頼する", JAPANESE);
      selection_demo.set_text(U"Επίδειξη Ask_Selection", GREEK);
      selection_demo.set_text(U"Dimostrazione di Ask_Selection", ITALIAN);
      selection_demo.set_text(U"Демонстрация функции Ask_Selection", RUSSIAN);
      selection_demo.set_text(U"הדגמת Ask_Selection", HEBREW);
    }
  
  draw_text(selection_demo,
	    WHITE,
	    width/2,
	    height*1/20,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  set_text_size(height/50);

  color c = GREY5;
  static multilingua txt1;
  if(init)
    {
      txt1.set_text("Selections can be made with a ask_selection control.", ENGLISH);
      txt1.set_text(U"可以使用 ask_selection 控件进行选择。", CHINESE_SIMPLE);
      txt1.set_text(U"可以使用 ask_selection 控制項進行選擇。", CHINESE);
      txt1.set_text(U"सिलेक्शन ask_selection कंट्रोल से किए जा सकते हैं।", HINDI);
      txt1.set_text(U"Las selecciones se pueden realizar con un control ask_selection.", SPANISH);
      txt1.set_text(U"Les sélections peuvent être effectuées à l'aide d'un contrôle ask_selection.", FRENCH);
      txt1.set_text(U"يمكن إجراء التحديدات باستخدام عنصر التحكم ask_selection.", ARABIC);
      txt1.set_text(U"একটি ask_selection কন্ট্রোল ব্যবহার করে নির্বাচন করা যেতে পারে।", BENGALI);
      txt1.set_text(U"Вибір можна зробити за допомогою елемента керування ask_selection.", UKRANIAN);
      txt1.set_text(U"É possível fazer seleções com um controlo ask_selection.", PORTUGUESE);
      txt1.set_text(U"اسک_سلیکشن کنٹرول کے ساتھ انتخاب کیے جا سکتے ہیں۔", URDU);
      txt1.set_text(U"Pemilihan dapat dilakukan dengan menggunakan kontrol ask_selection.", INDONESIAN);
      txt1.set_text(U"Die Auswahl kann mit einem ask_selection-Steuerelement getroffen werden.", GERMAN);
      txt1.set_text(U"選択はask_selectionコントロールを使用して行うことができます。", JAPANESE);
      txt1.set_text(U"Οι επιλογές μπορούν να γίνουν με ένα στοιχείο ελέγχου ask_selection.", GREEK);
      txt1.set_text(U"È possibile effettuare selezioni tramite il controllo ask_selection.", ITALIAN);
      txt1.set_text(U"Выбор можно сделать с помощью элемента управления ask_selection.", RUSSIAN);
      txt1.set_text(U"ניתן לבצע בחירות באמצעות פקד ask_selection.", HEBREW);
    }
  
  draw_text(txt1,
	    c,
	    width/2,
	    height*11/40,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);


  static multilingua txt1a;
  if(init)
    {
      txt1a.set_text("Selection controls use no extra screen space.", ENGLISH);
      txt1a.set_text(U"选择控件不占用额外的屏幕空间。", CHINESE_SIMPLE);
      txt1a.set_text(U"選擇控制不佔用額外的螢幕空間。", CHINESE);
      txt1a.set_text(U"सिलेक्शन कंट्रोल कोई एक्स्ट्रा स्क्रीन स्पेस इस्तेमाल नहीं करते हैं।", HINDI);
      txt1a.set_text(U"Los controles de selección no utilizan espacio adicional en la pantalla.", SPANISH);
      txt1a.set_text(U"Les commandes de sélection n'utilisent aucun espace d'écran supplémentaire.", FRENCH);
      txt1a.set_text(U"لا تستخدم عناصر التحكم في التحديد أي مساحة إضافية على الشاشة.", ARABIC);
      txt1a.set_text(U"নির্বাচনের জন্য ব্যবহৃত নিয়ন্ত্রণগুলো কোনো অতিরিক্ত স্ক্রিন স্পেস ব্যবহার করে না।", BENGALI);
      txt1a.set_text(U"Елементи керування вибором не використовують додатковий простір на екрані.", UKRANIAN);
      txt1a.set_text(U"Os controlos de seleção não ocupam espaço adicional no ecrã.", PORTUGUESE);
      txt1a.set_text(U"سلیکشن کنٹرولز اسکرین کی کوئی اضافی جگہ استعمال نہیں کرتے ہیں۔", URDU);
      txt1a.set_text(U"Kontrol pemilihan tidak menggunakan ruang layar tambahan.", INDONESIAN);
      txt1a.set_text(U"Die Auswahlsteuerlemente benötigen keinen zusätzlichen Bildschirmplatz.", GERMAN);
      txt1a.set_text(U"選択コントロールは追加の画面スペースを必要としません。", JAPANESE);
      txt1a.set_text(U"Τα στοιχεία ελέγχου επιλογής δεν χρησιμοποιούν επιπλέον χώρο στην οθόνη.", GREEK);
      txt1a.set_text(U"I controlli di selezione non occupano spazio aggiuntivo sullo schermo.", ITALIAN);
      txt1a.set_text(U"Элементы управления выбором не занимают дополнительного места на экране.", RUSSIAN);
      txt1a.set_text(U"פקדי בחירה אינם משתמשים בשטח מסך נוסף.", HEBREW);
    }
  
  draw_text(txt1a,
	    c,
	    width/2,
	    height*13/40,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  
  static multilingua txt2;
  if(init)
    {
      txt2.set_text("All other actions are blocked while a selection is made.", ENGLISH);
      txt2.set_text(U"在进行选择时，所有操作都会被阻止。", CHINESE_SIMPLE);
      txt2.set_text(U"進行選擇時，其他操作將被阻止。", CHINESE);
      txt2.set_text(U"सिलेक्शन होने पर एक्शन ब्लॉक हो जाते हैं।", HINDI);
      txt2.set_text(U"Las acciones están bloqueadas mientras se realiza una selección.", SPANISH);
      txt2.set_text(U"Les actions sont bloquées pendant qu'une sélection est effectuée.", FRENCH);
      txt2.set_text(U"يتم حظر الإجراءات أثناء عملية التحديد.", ARABIC);
      txt2.set_text(U"নির্বাচন করা অবস্থায় কোনো কাজ করা যায় না।", BENGALI);
      txt2.set_text(U"Дії блокуються, поки виконується вибір.", UKRANIAN);
      txt2.set_text(U"As ações são bloqueadas enquanto é feita uma seleção.", PORTUGUESE);
      txt2.set_text(U"انتخاب کے دوران کارروائیاں مسدود ہیں۔", URDU);
      txt2.set_text(U"Tindakan akan diblokir selama proses pemilihan berlangsung.", INDONESIAN);
      txt2.set_text(U"Während eine Auswahl getroffen wird, sind Aktionen blockiert.", GERMAN);
      txt2.set_text(U"選択が行われている間は、操作はブロックされます。", JAPANESE);
      txt2.set_text(U"Οι ενέργειες αποκλείονται κατά την πραγματοποίηση μιας επιλογής.", GREEK);
      txt2.set_text(U"Le azioni sono bloccate durante la selezione.", ITALIAN);
      txt2.set_text(U"Во время выделения элементов выполнение действий заблокировано.", RUSSIAN);
      txt2.set_text(U"פעולות חסומות בזמן ביצוע בחירה.", HEBREW);
    }
  draw_text(txt2,
	    c,
	    width/2,
	    height*55/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua txt3;
  if(init)
    {
      txt3.set_text("Multiselect and select_one are similar.", ENGLISH);
      txt3.set_text(U"“多选”和“单选”功能类似。", CHINESE_SIMPLE);
      txt3.set_text(U"“多選”和“單選”類似。", CHINESE);
      txt3.set_text(U"'मल्टीसेलेक्ट' और 'सेलेक्ट_वन' एक जैसे हैं।", HINDI);
      txt3.set_text(U"Las opciones 'multiselect' y 'select_one' son similares.", SPANISH);
      txt3.set_text(U"Les options « multiselect » et « select_one » sont similaires.", FRENCH);
      txt3.set_text(U"تتشابه خيارات \"الاختيار المتعدد\" و\"الاختيار الفردي\".", ARABIC);
      txt3.set_text(U"'মাল্টিসিলেক্ট' এবং 'সিলেক্ট_ওয়ান' একই রকম।", BENGALI);
      txt3.set_text(U"«Мультивибір» та «вибір_одного» схожі.", UKRANIAN);
      txt3.set_text(U"'Multiselect' e 'select_one' são semelhantes.", PORTUGUESE);
      txt3.set_text(U"'Multiselect' اور 'select_one' ایک جیسے ہیں۔", URDU);
      txt3.set_text(U"'Multiselect' dan 'select_one' memiliki kemiripan.", INDONESIAN);
      txt3.set_text(U"„Multiselect“ und „select_one“ sind ähnlich.", GERMAN);
      txt3.set_text(U"「マルチセレクト」と「シングルセレクト」は似ています。", JAPANESE);
      txt3.set_text(U"Οι εντολές «Πολλαπλή επιλογή» και «επιλογή_μίας» είναι παρόμοιες.", GREEK);
      txt3.set_text(U"Le opzioni \"multiselect\" e \"select_one\" sono simili.", ITALIAN);
      txt3.set_text(U"Типы полей «multiselect» и «select_one» похожи.", RUSSIAN);
      txt3.set_text(U"'בחירה מרובה' ו-'בחירה_אחד' דומים.", HEBREW);
    }
  draw_text(txt3,
	    c,
	    width/2,
	    height*60/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  static multilingua txt4;
  if(init)
    {
      txt4.set_text("Yet multiselect and select_one use screen space.", ENGLISH);
      txt4.set_text(U"然而，“多选”和“单选”都会占用屏幕空间。", CHINESE_SIMPLE);
      txt4.set_text(U"然而，「多選」和「單選」會佔用螢幕空間。", CHINESE);
      txt4.set_text(U"फिर भी 'मल्टीसेलेक्ट' और 'सेलेक्ट_वन' स्क्रीन स्पेस का इस्तेमाल करते हैं।", HINDI);
      txt4.set_text(U"Sin embargo, las opciones \"selección múltiple\" y \"seleccionar una\" ocupan espacio en la pantalla.", SPANISH);
      txt4.set_text(U"Cependant, les options « multiselect » et « select_one » occupent de l'espace à l'écran.", FRENCH);
      txt4.set_text(U"مع ذلك، فإن خياري \"الاختيار المتعدد\" و\"الاختيار الفردي\" يشغلان مساحة على الشاشة.", ARABIC);
      txt4.set_text(U"তবে ‘মাল্টিসিলেক্ট’ এবং ‘সিলেক্ট_ওয়ান’ স্ক্রিনের জায়গা ব্যবহার করে।", BENGALI);
      txt4.set_text(U"Однак «multiselect» та «select_one» використовують простір екрану.", UKRANIAN);
      txt4.set_text(U"No entanto, 'multiselect' e 'select_one' utilizam espaço no ecrã.", PORTUGUESE);
      txt4.set_text(U"پھر بھی 'ملٹی سلیکٹ' اور 'سلیکٹ_ون' اسکرین کی جگہ استعمال کرتے ہیں۔", URDU);
      txt4.set_text(U"Namun, opsi 'multiselect' dan 'select_one' menggunakan ruang layar.", INDONESIAN);
      txt4.set_text(U"Doch die Optionen „Mehrfachauswahl“ und „Einzelauswahl“ benötigen Bildschirmplatz.", GERMAN);
      txt4.set_text(U"しかし、「複数選択」と「単一選択」は画面スペースを消費します。", JAPANESE);
      txt4.set_text(U"Ωστόσο, οι εντολές 'multiselect' και 'select_one' χρησιμοποιούν χώρο στην οθόνη.", GREEK);
      txt4.set_text(U"Tuttavia, le opzioni \"multiselect\" e \"select_one\" occupano spazio sullo schermo.", ITALIAN);
      txt4.set_text(U"Однако элементы управления «multiselect» и «select_one» занимают место на экране.", RUSSIAN);
      txt4.set_text(U"עם זאת, 'multiselect' ו-'select_one' משתמשים בשטח המסך.", HEBREW);
    }
  draw_text(txt4,
	    c,
	    width/2,
	    height*65/80,
	    DRAW_TEXT_X_CENTER|DRAW_TEXT_Y_TOP|DRAW_TEXT_ROTATE_0);

  init = false;
}


void button_push_function_questionable_action(button*b, void* v)
{
  selection_demo_window* bdw = (selection_demo_window*) v;

  display* d = b->get_display();

  static bool init = true;
  static multilingua question;  
  static multilingua answers[4];
  static multilingua end_text;

  if(init)
    {
      question.set_text("Are You Sure?", ENGLISH);
      question.set_text(U"你确定吗？", CHINESE_SIMPLE);
      question.set_text(U"你確定嗎？", CHINESE);
      question.set_text(U"क्या आपको यकीन है?", HINDI);
      question.set_text(U"¿Estas seguro?", SPANISH);
      question.set_text(U"Es-tu sûr?", FRENCH);
      question.set_text(U"هل أنت متأكد؟", ARABIC);
      question.set_text(U"আপনি কি নিশ্চিত?", BENGALI);
      question.set_text(U"Ви впевнені?", UKRANIAN);
      question.set_text(U"Tem a certeza?", PORTUGUESE);
      question.set_text(U"کیا آپ کو یقین ہے؟", URDU);
      question.set_text(U"Apa kamu yakin?", INDONESIAN);
      question.set_text(U"Bist du sicher?", GERMAN);
      question.set_text(U"本気ですか？", JAPANESE);
      question.set_text(U"Είσαι σίγουρος/η;", GREEK);
      question.set_text(U"Sei sicuro?", ITALIAN);
      question.set_text(U"Вы уверены?", RUSSIAN);
      question.set_text(U"האם אתה בטוח?", HEBREW);

      answers[0].set_text("Yes", ENGLISH);
      answers[0].set_text(U"是的", CHINESE_SIMPLE);
      answers[0].set_text(U"是的", CHINESE);
      answers[0].set_text(U"हाँ", HINDI);
      answers[0].set_text(U"Sí", SPANISH);
      answers[0].set_text(U"Oui", FRENCH);
      answers[0].set_text(U"نعم", ARABIC);
      answers[0].set_text(U"হ্যাঁ", BENGALI);
      answers[0].set_text(U"Так", UKRANIAN);
      answers[0].set_text(U"Sim", PORTUGUESE);
      answers[0].set_text(U"جی ہاں", URDU);
      answers[0].set_text(U"Ya", INDONESIAN);
      answers[0].set_text(U"Ja", GERMAN);
      answers[0].set_text(U"はい", JAPANESE);
      answers[0].set_text(U"Ναί", GREEK);
      answers[0].set_text(U"SÌ", ITALIAN);
      answers[0].set_text(U"Да", RUSSIAN);
      answers[0].set_text(U"כֵּן", HEBREW);

      answers[1].set_text("No", ENGLISH);
      answers[1].set_text(U"不", CHINESE_SIMPLE);
      answers[1].set_text(U"不", CHINESE);
      answers[1].set_text(U"नहीं", HINDI);
      answers[1].set_text(U"No", SPANISH);
      answers[1].set_text(U"Non", FRENCH);
      answers[1].set_text(U"لا", ARABIC);
      answers[1].set_text(U"না", BENGALI);
      answers[1].set_text(U"Ні", UKRANIAN);
      answers[1].set_text(U"Não", PORTUGUESE);
      answers[1].set_text(U"نہیں", URDU);
      answers[1].set_text(U"Tidak", INDONESIAN);
      answers[1].set_text(U"Nein", GERMAN);
      answers[1].set_text(U"いいえ", JAPANESE);
      answers[1].set_text(U"Οχι", GREEK);
      answers[1].set_text(U"No", ITALIAN);
      answers[1].set_text(U"Нет", RUSSIAN);
      answers[1].set_text(U"לֹא", HEBREW);

      answers[2].set_text("Perhaps", ENGLISH);
      answers[2].set_text(U"也许", CHINESE_SIMPLE);
      answers[2].set_text(U"也許", CHINESE);
      answers[2].set_text(U"शायद", HINDI);
      answers[2].set_text(U"Tal vez", SPANISH);
      answers[2].set_text(U"Peut-être", FRENCH);
      answers[2].set_text(U"ربما", ARABIC);
      answers[2].set_text(U"সম্ভবত", BENGALI);
      answers[2].set_text(U"Можливо", UKRANIAN);
      answers[2].set_text(U"Talvez", PORTUGUESE);
      answers[2].set_text(U"شاید", URDU);
      answers[2].set_text(U"Mungkin", INDONESIAN);
      answers[2].set_text(U"Vielleicht", GERMAN);
      answers[2].set_text(U"多分", JAPANESE);
      answers[2].set_text(U"Ισως", GREEK);
      answers[2].set_text(U"Forse", ITALIAN);
      answers[2].set_text(U"Возможно", RUSSIAN);
      answers[2].set_text(U"אוּלַי", HEBREW);

      answers[3].set_text("Undecided", ENGLISH);
      answers[3].set_text(U"尚未决定", CHINESE_SIMPLE);
      answers[3].set_text(U"未決定", CHINESE);
      answers[3].set_text(U"कच्चा पक्का", HINDI);
      answers[3].set_text(U"Indeciso", SPANISH);
      answers[3].set_text(U"Indécis", FRENCH);
      answers[3].set_text(U"غير محدد", ARABIC);
      answers[3].set_text(U"অনির্ধারিত", BENGALI);
      answers[3].set_text(U"Не визначився", UKRANIAN);
      answers[3].set_text(U"Indeciso", PORTUGUESE);
      answers[3].set_text(U"غیر فیصلہ کن", URDU);
      answers[3].set_text(U"Bimbang", INDONESIAN);
      answers[3].set_text(U"Unentschieden", GERMAN);
      answers[3].set_text(U"未定", JAPANESE);
      answers[3].set_text(U"Αναποφάσιστος", GREEK);
      answers[3].set_text(U"Indeciso", ITALIAN);
      answers[3].set_text(U"Неопределенный", RUSSIAN);
      answers[3].set_text(U"לֹא הֶחלֵטִי", HEBREW);

      end_text.set_text("Button was pushed with result \"", ENGLISH);
      end_text.set_text(U"按钮被按下，并产生了结果。 \"", CHINESE_SIMPLE);
      end_text.set_text(U"按鈕被按下，結果 \"", CHINESE);
      end_text.set_text(U"बटन दबाया गया और परिणाम मिला \"", HINDI);
      end_text.set_text(U"Se pulsó el botón y se obtuvo un resultado. \"", SPANISH);
      end_text.set_text(U"Le bouton a été appuyé et un résultat a été obtenu. \"", FRENCH);
      end_text.set_text(U"تم الضغط على الزر وكانت هذه هي النتيجة. \"", ARABIC);
      end_text.set_text(U"বোতাম চাপার ফলে এই ফলাফল পাওয়া গেল। \"", BENGALI);
      end_text.set_text(U"Кнопку натиснули, і результат був \"", UKRANIAN);
      end_text.set_text(U"O botão foi pressionado com o resultado \"", PORTUGUESE);
      end_text.set_text(U"نتیجہ کے ساتھ بٹن دبایا گیا۔ \"", URDU);
      end_text.set_text(U"Tombol telah ditekan dan menghasilkan suatu hasil. \"", INDONESIAN);
      end_text.set_text(U"Der Knopf wurde gedrückt, mit folgendem Ergebnis: \"", GERMAN);
      end_text.set_text(U"ボタンが押され、結果が表示されました。 \"", JAPANESE);
      end_text.set_text(U"Το κουμπί πατήθηκε με αποτέλεσμα \"", GREEK);
      end_text.set_text(U"Il pulsante è stato premuto con successo. \"", ITALIAN);
      end_text.set_text(U"Кнопка была нажата, и получен результат. \"", RUSSIAN);
      end_text.set_text(U"כפתור נלחץ עם תוצאה \"", HEBREW);
    }

  const char32_t* answers_txt[4];
  const color TEXT_COLORS[4] = { GREEN7, GREEN7, GREEN7, GREEN7 };
  
  for(int i=0; i<4; i++)
    answers_txt[i] = answers[i];
  
  int answer = ask_selection(d,
			     question,
			     4,
			     answers_txt,
			     GREY4, // BG_COLOR,
			     RED,   // TITLE_COLOR,
			     GREY6, // BUTTON_COLOR,
			     TEXT_COLORS);

  //printf("Are you sure answer was %d\n", answer);
  
  bdw->last_text.set_text((const char32_t*)end_text);
  bdw->last_text.append_text((const char32_t*)answers[answer]);
  bdw->last_text.append_text("\".");
  bdw->last_text_count = bdw->text_display_interval;

  init = false;
}



void button_push_function_favorite_color(button*b, void* v)
{
  //  selection_demo_window* bdw = (selection_demo_window*) v;

  display* d = b->get_display();

  static bool init = true;
  static multilingua question;  
  static multilingua answers[4];
  static multilingua end_text;
  
  if(!favorite_colors)
    {
      favorite_colors = new multilingua[NUM_FAVS];
  
      favorite_colors[0].set_text("Red", ENGLISH);
      favorite_colors[0].set_text(U"红色的", CHINESE_SIMPLE);
      favorite_colors[0].set_text(U"紅色的", CHINESE);
      favorite_colors[0].set_text(U"लाल", HINDI);
      favorite_colors[0].set_text(U"Roja", SPANISH);
      favorite_colors[0].set_text(U"Rouge", FRENCH);
      favorite_colors[0].set_text(U"أحمر", ARABIC);
      favorite_colors[0].set_text(U"লাল", BENGALI);
      favorite_colors[0].set_text(U"Червоний", UKRANIAN);
      favorite_colors[0].set_text(U"Vermelha", PORTUGUESE);
      favorite_colors[0].set_text(U"سرخ", URDU);
      favorite_colors[0].set_text(U"Merah", INDONESIAN);
      favorite_colors[0].set_text(U"Rot", GERMAN);
      favorite_colors[0].set_text(U"赤", JAPANESE);
      favorite_colors[0].set_text(U"Κόκκινος", GREEK);
      favorite_colors[0].set_text(U"Rossa", ITALIAN);
      favorite_colors[0].set_text(U"Красный", RUSSIAN);
      favorite_colors[0].set_text(U"אָדוֹם", HEBREW);
      
      favorite_colors[1].set_text("Green", ENGLISH);
      favorite_colors[1].set_text(U"绿色的", CHINESE_SIMPLE);
      favorite_colors[1].set_text(U"綠色的", CHINESE);
      favorite_colors[1].set_text(U"हरा", HINDI);
      favorite_colors[1].set_text(U"Verde", SPANISH);
      favorite_colors[1].set_text(U"Verte", FRENCH);
      favorite_colors[1].set_text(U"أخضر", ARABIC);
      favorite_colors[1].set_text(U"সবুজ", BENGALI);
      favorite_colors[1].set_text(U"Зелений", UKRANIAN);
      favorite_colors[1].set_text(U"Verde", PORTUGUESE);
      favorite_colors[1].set_text(U"سبز", URDU);
      favorite_colors[1].set_text(U"Hijau", INDONESIAN);
      favorite_colors[1].set_text(U"Grün", GERMAN);
      favorite_colors[1].set_text(U"緑", JAPANESE);
      favorite_colors[1].set_text(U"Πράσινος", GREEK);
      favorite_colors[1].set_text(U"Verde", ITALIAN);
      favorite_colors[1].set_text(U"Зеленый", RUSSIAN);
      favorite_colors[1].set_text(U"יָרוֹק", HEBREW);
      
      favorite_colors[2].set_text("Blue", ENGLISH);
      favorite_colors[2].set_text(U"蓝色的", CHINESE_SIMPLE);
      favorite_colors[2].set_text(U"藍色的", CHINESE);
      favorite_colors[2].set_text(U"नीला", HINDI);
      favorite_colors[2].set_text(U"Azul", SPANISH);
      favorite_colors[2].set_text(U"Bleue", FRENCH);
      favorite_colors[2].set_text(U"أزرق", ARABIC);
      favorite_colors[2].set_text(U"নীল", BENGALI);
      favorite_colors[2].set_text(U"Синій", UKRANIAN);
      favorite_colors[2].set_text(U"Azul", PORTUGUESE);
      favorite_colors[2].set_text(U"نیلا", URDU);
      favorite_colors[2].set_text(U"Biru", INDONESIAN);
      favorite_colors[2].set_text(U"Blau", GERMAN);
      favorite_colors[2].set_text(U"青", JAPANESE);
      favorite_colors[2].set_text(U"Μπλε", GREEK);
      favorite_colors[2].set_text(U"Blu", ITALIAN);
      favorite_colors[2].set_text(U"Синий", RUSSIAN);
      favorite_colors[2].set_text(U"כְּחוֹל", HEBREW);
      
      favorite_colors[3].set_text("Yellow", ENGLISH);
      favorite_colors[3].set_text(U"黄色的", CHINESE_SIMPLE);
      favorite_colors[3].set_text(U"黃色的", CHINESE);
      favorite_colors[3].set_text(U"पीला", HINDI);
      favorite_colors[3].set_text(U"Amarilla", SPANISH);
      favorite_colors[3].set_text(U"Jaune", FRENCH);
      favorite_colors[3].set_text(U"أصفر", ARABIC);
      favorite_colors[3].set_text(U"হলুদ", BENGALI);
      favorite_colors[3].set_text(U"Жовтий", UKRANIAN);
      favorite_colors[3].set_text(U"Amarela", PORTUGUESE);
      favorite_colors[3].set_text(U"پیلا", URDU);
      favorite_colors[3].set_text(U"Kuning", INDONESIAN);
      favorite_colors[3].set_text(U"Gelb", GERMAN);
      favorite_colors[3].set_text(U"黄色", JAPANESE);
      favorite_colors[3].set_text(U"Κίτρινος", GREEK);
      favorite_colors[3].set_text(U"Gialla", ITALIAN);
      favorite_colors[3].set_text(U"Желтый", RUSSIAN);
      favorite_colors[3].set_text(U"צָהוֹב", HEBREW);
      
      favorite_colors[4].set_text("Orange", ENGLISH);
      favorite_colors[4].set_text(U"橙子", CHINESE_SIMPLE);
      favorite_colors[4].set_text(U"橘子", CHINESE);
      favorite_colors[4].set_text(U"नारंगी", HINDI);
      favorite_colors[4].set_text(U"Naranja", SPANISH);
      favorite_colors[4].set_text(U"Orange", FRENCH);
      favorite_colors[4].set_text(U"البرتقالي", ARABIC);
      favorite_colors[4].set_text(U"কমলা", BENGALI);
      favorite_colors[4].set_text(U"Помаранчевий", UKRANIAN);
      favorite_colors[4].set_text(U"Laranja", PORTUGUESE);
      favorite_colors[4].set_text(U"کینو", URDU);
      favorite_colors[4].set_text(U"Oranye", INDONESIAN);
      favorite_colors[4].set_text(U"Orange", GERMAN);
      favorite_colors[4].set_text(U"オレンジ", JAPANESE);
      favorite_colors[4].set_text(U"Πορτοκάλι", GREEK);
      favorite_colors[4].set_text(U"Arancia", ITALIAN);
      favorite_colors[4].set_text(U"Апельсин", RUSSIAN);
      favorite_colors[4].set_text(U"כָּתוֹם", HEBREW);
      
      favorite_colors[5].set_text("Purple", ENGLISH);
      favorite_colors[5].set_text(U"紫色的", CHINESE_SIMPLE);
      favorite_colors[5].set_text(U"紫色的", CHINESE);
      favorite_colors[5].set_text(U"बैंगनी", HINDI);
      favorite_colors[5].set_text(U"Púrpura", SPANISH);
      favorite_colors[5].set_text(U"Violette", FRENCH);
      favorite_colors[5].set_text(U"أرجواني", ARABIC);
      favorite_colors[5].set_text(U"বেগুনি", BENGALI);
      favorite_colors[5].set_text(U"Фіолетовий", UKRANIAN);
      favorite_colors[5].set_text(U"Roxa", PORTUGUESE);
      favorite_colors[5].set_text(U"جامنی", URDU);
      favorite_colors[5].set_text(U"Ungu", INDONESIAN);
      favorite_colors[5].set_text(U"Lila", GERMAN);
      favorite_colors[5].set_text(U"紫", JAPANESE);
      favorite_colors[5].set_text(U"Μωβ", GREEK);
      favorite_colors[5].set_text(U"Viola", ITALIAN);
      favorite_colors[5].set_text(U"Фиолетовый", RUSSIAN);
      favorite_colors[5].set_text(U"סָגוֹל", HEBREW);
    }

  if(init)
    {
      question.set_text("Which is your favorite?", ENGLISH);
      question.set_text(U"你最喜欢哪一个？", CHINESE_SIMPLE);
      question.set_text(U"你最喜歡哪一個？", CHINESE);
      question.set_text(U"आपका पसंदीदा कौन सा है?", HINDI);
      question.set_text(U"¿Cuál es tu favorito?", SPANISH);
      question.set_text(U"Lequel est votre préféré ?", FRENCH);
      question.set_text(U"ما هو المفضل لديك؟", ARABIC);
      question.set_text(U"আপনার প্রিয় কোনটি?", BENGALI);
      question.set_text(U"Який ваш улюблений?", UKRANIAN);
      question.set_text(U"Qual é o seu favorito?", PORTUGUESE);
      question.set_text(U"آپ کا پسندیدہ کون سا ہے؟", URDU);
      question.set_text(U"Mana yang menjadi favoritmu?", INDONESIAN);
      question.set_text(U"Welches ist dein Favorit?", GERMAN);
      question.set_text(U"あなたのお気に入りはどれですか？", JAPANESE);
      question.set_text(U"Ποιο είναι το αγαπημένο σου;", GREEK);
      question.set_text(U"Qual è il tuo preferito?", ITALIAN);
      question.set_text(U"Какой из них ваш любимый?", RUSSIAN);
      question.set_text(U"איזה מהם האהוב עליך?", HEBREW);
    }

  const char32_t* answers_txt[NUM_FAVS];
  
  for(int i=0; i<NUM_FAVS; i++)
    answers_txt[i] = favorite_colors[i];
  
  int answer = ask_selection(d,
			     question,
			     NUM_FAVS,
			     answers_txt,
			     GREY4, // BG_COLOR,
			     WHITE, // TITLE_COLOR,
			     GREY6, // BUTTON_COLOR,
			     &fav_colors.c[1]);

  //printf("Color answer was %d\n", answer);

  static multilingua favcoloris;
  if(init)
    {
      favcoloris.set_text("Your favorite colour is", ENGLISH);
      favcoloris.set_text(U"你最喜欢的颜色是", CHINESE_SIMPLE);
      favcoloris.set_text(U"你最喜歡的顏色是", CHINESE);
      favcoloris.set_text(U"आपका पसंदीदा रंग है", HINDI);
      favcoloris.set_text(U"Tu color favorito es", SPANISH);
      favcoloris.set_text(U"Votre couleur préférée est", FRENCH);
      favcoloris.set_text(U"لونك المفضل هو", ARABIC);
      favcoloris.set_text(U"আপনার প্রিয় রঙ হলো", BENGALI);
      favcoloris.set_text(U"Твій улюблений колір", UKRANIAN);
      favcoloris.set_text(U"A sua cor favorita é", PORTUGUESE);
      favcoloris.set_text(U"آپ کا پسندیدہ رنگ ہے۔", URDU);
      favcoloris.set_text(U"Warna favoritmu adalah", INDONESIAN);
      favcoloris.set_text(U"Deine Lieblingsfarbe ist", GERMAN);
      favcoloris.set_text(U"あなたの好きな色は", JAPANESE);
      favcoloris.set_text(U"Το αγαπημένο σου χρώμα είναι", GREEK);
      favcoloris.set_text(U"Il tuo colore preferito è", ITALIAN);
      favcoloris.set_text(U"Ваш любимый цвет —", RUSSIAN);
      favcoloris.set_text(U"הצבע האהוב עליך הוא", HEBREW);
    }

  for(int language=0; language<NUM_LANGUAGES; language++)
    {
      unalingua bname;

      bname.set_text(favcoloris.get_text(language));

      char32_t txt[3];
      txt[0] = ' '; 
      txt[1] = 10 + 1 + answer;
      txt[2] = 0;

      bname.append_text(txt);
  
      bname.append_text(favorite_colors[answer].get_text(language));
      b->set_name((const char32_t*) bname, language);
    }
  
  init = false;
}
